"""
:synopsis: Compare NG911 Feature Classes
:authors: Riley Baird (OK), Emma Baker (OK)
"""
from collections.abc import Sequence

import arcpy

from ...lib.comparison import NG911GeodatabaseComparator
from ...lib.misc import unwrap_to_dict
from ...lib.session import config


class FeatureClassCompareTool:
    """
    Class describing an ArcGIS Python Toolbox tool: "Compare NG911 Feature Classes".
    """

    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "Compare NG911 Feature Classes"
        self.description = ""
        self.canRunInBackground = False
        self.category = "5 - Comparison"

    def getParameterInfo(self):
        """Define parameter definitions"""
        params = [
            arcpy.Parameter(
                displayName="Geodatabase 1",
                name="gdb1",
                datatype="DEWorkspace",
                parameterType="Required",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Geodatabase 2",
                name="gdb2",
                datatype="DEWorkspace",
                parameterType="Required",
                direction="Input"
            ),
            feature_class := arcpy.Parameter(
                displayName="Feature Class",
                name="feature_class",
                datatype="GPString",
                parameterType="Required",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Respect Submit",
                name="respect_submit",
                datatype="GPBoolean",
                parameterType="Optional",
                direction="Input"
            ),
            arcpy.Parameter(
                displayName="Export Results to Geodatabase 1",
                name="export",
                datatype="GPBoolean",
                parameterType="Optional",
                direction="Input"
            ),
            # OUTPUT
            arcpy.Parameter(
                displayName="Comparison Results Table",
                name="comparison_results_table",
                datatype="DETable",
                parameterType="Derived",
                direction="Output"
            )
        ]
        feature_class.filter.list = list(config.required_feature_class_names + config.optional_feature_class_names)
        return params

    def isLicensed(self):
        """Set whether tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter.  This method is called after internal validation."""
        return

    def execute(self, parameters: Sequence[arcpy.Parameter], messages):
        """The source code of the tool."""
        param_values = unwrap_to_dict(parameters)
        comparator = NG911GeodatabaseComparator(
            param_values["gdb1"],
            param_values["gdb2"],
            bool(param_values["respect_submit"])
        )
        fc_name = param_values["feature_class"]
        fc = config.get_feature_class_by_name(fc_name)
        if param_values["export"]:
            out_table_name = f"ComparisonResults_{fc_name}"
            comparator.compare_feature_class(fc, comparator.gdb1.gdb_path, out_table_name)
            parameters[-1].value = str(comparator.gdb1.gdb_path / out_table_name)
        else:
            comparator.compare_feature_class(fc)


if __name__ == "__main__":
    raise Exception("This module is a dependency of an ArcGIS Python Toolbox and should not be executed directly.")
